﻿(function($, ns, undefined) {
    var application = Backbone.Model.extend({
        //properties
        //message postback to
        msgReceiver: null,
        msgOrigin: null,
        //attributes
        defaults: {
            //tracks if the current user is global admin
            currentUserRoles: 0,
            //tracks the previous state of the library search
            previousState: null,
            //tracks the current user
            currentUserName: null
        },
        //functions
        initialize: function() {
            var self = this;

            //listen to postMessages
            if (window.addEventListener) {
                window.addEventListener("message", function(msg) { self.messageReceiver(msg); }, false);
            } else if (window.attachEvent) {
                window.attachEvent("message", function(msg) { self.messageReceiver(msg); });
            } else {
                throw "Cannot attachEvent/addEventListener, not supported by browser";
            }

            //pass true to use alert as fallback for the logger.
            this.setupLogger();
            var virtualPath = this.get('virtualPath');
            if (!virtualPath)
                throw "Missing virtualPath from configuration. Please pass this as an argument when creating the application";
            if (!MeridiumUtil.endsWith(virtualPath, "/"))
                this.set({ "virtualPath": virtualPath + "/" });
            //add logging of all events 
            //TODO:disable this in runtime....
            this.bind("all", function() {
                //self.log("Application event " + Array.prototype.join.call(arguments, ","));
            });
        },
        getPageLang: function() {
            return document.documentElement.lang || null;
        },
        //listens to postMessage messages and sets up a response channel
        messageReceiver: function(msg) {
            if (msg.data === "init") {
                this.msgReceiver = msg.source;
                this.msgTargetOrigin = msg.origin;
                msg.source.postMessage("initReceived", msg.origin);
                //'this.trigger' doesn't work?!
                this.trigger("app:openedForInsertChanged");
                return;
            }
            //keepalive signal (replaces init, init is kept for backwards compability)
            if (msg.data === "ping") {
                if (!this.msgReceiver) {
                    this.msgReceiver = msg.source;
                    this.msgTargetOrigin = msg.origin;
                    msg.source.postMessage("pong", msg.origin);
                    //'this.trigger' doesn't work?!
                    this.trigger("app:openedForInsertChanged");
                    return;
                }
            }
        },
        //checks if we are opened for insert
        openedForInsert: function() {
            try {
                return (this.msgReceiver && this.msgTargetOrigin) || (opener && opener.IVCallback);
            } catch (e) {
                return false;
            }
        },

        // Close window
        closeWindow: function() {
            if (this.msgReceiver) {
                this.msgReceiver.postMessage("close", this.msgTargetOrigin);
            }
        },

        //sends a reply on a insert completion and closes the window.
        insertCallback: function(data) {
            if (this.msgReceiver && this.msgTargetOrigin) {
                if (data != null) {
                    //Json-serialize the returned object/s.
                    data = JSON.stringify(data);
                    this.msgReceiver.postMessage(data, this.msgTargetOrigin);
                }
                this.closeWindow();

            } else if (opener && opener.IVCallback) {
                //Json-serialize the returned object/s.
                data = JSON.stringify(data);
                opener.IVCallback(window, data);
                window.parent.close();
            } else {
                throw "Unable to call insert callback.";
            }
        },
        //checks if the current user is in the supplied role (or one of the supplied roles).
        //See ImageVaultUi.GlobalRoles for valid roles to check
        //@param role	the role(s) to check
        //@return	true if the user is part of the role(s)
        isUserInRole: function(role) {
            return (this.get("currentUserRoles") & role) > 0;
        },
        //the logger used by the application
        //fixes the console.log function and makes sure that it exists.
        setupLogger: function() {
            //ie will throw an error if the console window isn't open when accessing the console symbol
            try {
                if (console && console.log) {
                    application.prototype.log = console.log.bind(window.console);
                }
            } catch (e) {}
        },
        //logs a message to the application (dummy implementation, connected to console.log using setupLogger)
        log: function() {},
        //checks if the state has changed
        hasStateChanged: function(key) {
            var current = $.bbq.getState(key);
            var previousState = this.get("previousState");
            var previous;

            if (previousState) {
                previous = previousState.getState(key);
            }
            if (current === previous) {
                return false;
            }
            return true;
        },
        //Parses the supplied arguments and produces a config object 
        //config: base configuration for the method
        //controller: first argument, either the name of the controller or a confi object
        //action: second argument, either the name of the action or the success callback
        parseArguments: function(config, controller, action, data, success) {
            //parse controller
            if (typeof controller === "object") {
                $.extend(config, controller);
            } else {
                config.controller = controller;
            }
            //parse action
            if (action !== undefined) {
                if (typeof action === "string") {
                    config.action = action;
                } else if (typeof action === "object") {
                    config.data = action;
                } else if (typeof action === "function") {
                    config.success = action;
                }
                //parse data
                if (data !== undefined) {
                    if (typeof data === "function") {
                        config.success = data;
                    } else {
                        config.data = data;
                    }
                    //parse success
                    if (typeof success === "function") {
                        config.success = success;
                    }
                }
            }
            //fix relative url:s into controller and action
            if (config.url !== undefined) {
                //check for relative url:s
                if (config.url.indexOf("://") == -1) {
                    var pos = config.url.indexOf("/");
                    //url:controller
                    if (pos == -1) {
                        config.controller = config.url;
                        config.url = undefined;
                        //url: controller/action
                    } else if (pos > 0) {
                        config.controller = config.url.substr(0, pos);
                        config.action = config.url.substr(pos + 1);
                        config.url = undefined;
                    }
                }
            }

            //Calculate url if it is missing
            if (config.url === undefined)
                config.url = this.createPath(config.controller, config.action);
            //stringify data if we haven't already a contentType and it is an object
            if (!config.contentType && typeof config.data === "object") {
                config.data = JSON.stringify(config.data);
            }

            //custom error handling
            if (config.error) {
                config.customError = config.error;
                config.error = undefined;
            }
            return config;
        },

        //performs an ajax call that accepts html as response
        load: function(controller, action, data, success, $buttonsToDisable) {
            var config = { dataType: 'HTML' };
            config = this.parseArguments(config, controller, action, data, success, $buttonsToDisable);
            this.ajax(config);
        },
        //performs an ajax call that accepts json as response
        json: function(controller, action, data, success) {
            var config = { dataType: 'JSON' };
            config = this.parseArguments(config, controller, action, data, success);
            this.ajax(config);
        },
        getJson: function(controller, action, success) {
            var config = { type: "GET", dataType: 'JSON' };
            config = this.parseArguments(config, controller, action, null, success);
            this.ajax(config);
        },
        //posts a form with query string paramters as data
        form: function(controller, action, data, success) {
            var config = {
                type: 'POST',
                contentType: 'application/x-www-form-urlencoded',
                dataType: 'JSON'
            };
            config = this.parseArguments(config, controller, action, data, success);
            this.ajax(config);
        },
        //creates the virtual path to the supplied action of the controller
        createPath: function(controller, action) {

            var path = this.get('virtualPath');
            if (controller)
                path += controller;
            if (action)
                path += "/" + action;
            return path;
        },
        //wraps the ajax call and handles authentication errors and tries to reauthenticate the call if it fails
        ajax: function(config) {

            var self = this;
            config.error = function(jqXhr, status, err) {
                //if readystate is 0 then the request is aborted, probably due to a security issue, redirect blocked by CORS, etc.
                if (!jqXhr.readyState) {
                    self.showExpired();
                    return;
                }
                //if readystate is 4 then the request is complete
                //otherwise it has been aborted prematurely
                if (jqXhr.readyState !== 4)
                    return;
                //parse error response
                var errorResponse = null;
                try {
                    errorResponse = JSON.parse(jqXhr.responseText);
                } catch (e) {}
                if (jqXhr.status === 401) {
                    //if we get a 401 (probably a session timed out) we create an iframe and try to load the entry page in it.
                    //alert("Authentication failed, need to renew token for url '" + config.url + "': " + err+". Wait one second and try again...");
                    var i = document.createElement('iframe');
                    i.style.display = 'none';
                    //go to a page that requires authentication, this will force the browser to complete the passive authentication once more.
                    //Todo:This path don't work with ImageVault installations that isn't installed on the /ImageVault/ path...
                    var pathname = self.createPath("Account", "Authenticate");
                    i.src = pathname;
                    i.onload = function() {
                        //the passive login procedure will redirect the document to the ticket authenticator and after a few redirects
                        //we will finally end up at the requested location. Then we can remove the iframe and perform the ajax request again.
                        if (i.contentWindow.document.location.pathname == pathname) {
                            i.parentNode.removeChild(i);
                            config.error = null;
                            self.innerAjax(config);
                        }
                        //TODO:timeout?
                    };
                    document.body.appendChild(i);
                } else {
                    var errMessage = (err && err.message ? err.message : err);
                    if (typeof config.customError == "function") {
                        config.customError(errorResponse || errMessage);
                    } else {
                        self.showError("Failure in ajax call to url '" + config.url + "': " + errMessage, errorResponse);
                    }
                }
            };

            // Disable buttons while executing ajax request
            if (config.$buttonsToDisable !== undefined) {
                var loading = new IVNotification(translate("common.Loading"), { showOverlay: true });

                // Return if buttons already has disabled attribute
                if (config.$buttonsToDisable.attr("disabled") === "disabled")
                    return false;

                // First disable buttons
                config.$buttonsToDisable.attr("disabled", "disabled");

                // Then enable buttons again after success or error
                var successFunction = config.success;
                config.success = function() {
                    // Enable buttons
                    config.$buttonsToDisable.removeAttr("disabled");

                    // Remove loader
                    loading.remove();

                    // Original function

                    successFunction();

                }

                var errorFunction = config.error;
                config.error = function() {
                    // Enable buttons
                    config.$buttonsToDisable.removeAttr("disabled");

                    // Remove loader
                    loading.remove();

                    // Original function
                    errorFunction();
                }
            }

            this.innerAjax(config);
        },
        showError: function(errMessage, errorResponse) {
            var message = errMessage;
            //display error response from server if present
            if (errorResponse) {
                var detailedMessage = "";
                if (typeof(errorResponse) === "string") {
                    detailedMessage = errorResponse;
                } else if (errorResponse.message) {
                    detailedMessage = errorResponse.message;
                }
                if (message)
                    message += " \n \n" + detailedMessage;
            }
            alert(message);
        },
        showExpired: function() {
            IVNotification.removeAll();
            if ($("#expired-dialog").length) {
                return;
            }
            $("body").append('<div id="expired-dialog" class="overlay"><div class="dialog"><h1>' + translate('common.SessionExpired') + '</h1><p>' + translate('common.SessionExpiredMessage') + '</p><section class="actions"><a class="refresh-button blue-button" href="' + window.location.href + '">' + translate('common.Refresh') + '</a></section></div></div>');
            $("#expired-dialog a.refresh-button").click(function() {
                window.location.reload(true);
            });
        },
        //adjust config parameters and performs the actual ajax call
        innerAjax: function(config) {
            var self = this;
            //default error behaviour
            if (!config.error) {
                config.error = function(jqXhr, status, err) {
                    //if readystate is 4 then the request is complete
                    //otherwise it has been aborted prematurely
                    if (jqXhr.readystate != 4)
                        return;
                    alert("Failure in ajax call to url '" + config.url + "': " + err);
                };
            }
            //We expect that the returnvalue is html (if no other is supplied)
            if (!config.dataType)
                config.dataType = 'html';
            //Default to post
            if (!config.type)
                config.type = 'POST';
            //We send json data
            if (!config.contentType)
                config.contentType = 'application/json; charset=utf-8';

            var successFunction = config.success;
            config.success = function(r) {
                try {
                    successFunction(r);
                } catch (err) {

                    var errMessage = (err && err.message ? err.message : err);
                    if (typeof config.customError == "function") {
                        config.customError(errMessage);
                    } else {
                        self.showError("Failure in processing result of call: " + errMessage);
                    }
                }
            }

            let csrfToken = $("input[name='__ApiRequestVerificationToken']").val();
            if (csrfToken == undefined) csrfToken = '';

            config.headers = config.headers ? config.headers : {}
            config.headers['__ApiRequestVerificationToken'] = csrfToken;

            var xhr = $.ajax(config);
            //app.log(xhr);
        },
        // Checks if flash is installed
        hasFlash: function() {
            try {
                var fo = new ActiveXObject('ShockwaveFlash.ShockwaveFlash');
                if (fo) {
                    return true;
                }
            } catch (e) {
                if (navigator.mimeTypes && navigator.mimeTypes['application/x-shockwave-flash'] && navigator.mimeTypes['application/x-shockwave-flash'].enabledPlugin) {
                    return true;
                }
            }
            return false;
        }

    });
    ns.Application = application;
}(jQuery, window.ImageVaultUi = window.ImageVaultUi || {}));