﻿(function($, ns) {
    var imageVaultEditor = function(mediaId, formatId, loader, initConfig) {
        this.init(mediaId, formatId, loader, initConfig);
        return this;
    };

    imageVaultEditor.prototype = {
        targetFormat: null,
        format: null,
        client: null,
        media: null,
        loader: null,
        initConfig: {
            callback: null,
            editorOkButtonText: null,
            targetFormatId: null
        },

        //
        // Init function
        //
        init: function(mediaId, formatId, loader, initConfig) {
            var self = this;

            self.loader = loader;
            self.initConfig = initConfig ? initConfig : self.initConfig;

            //Gets the client for communicating with core
            if (!self.client) {
                self.client = new ImageVault.Client({
                    core: "/api",
                    authMethod: "none"
                });
            }

            // Get media to edit
            self.getMedia(mediaId, formatId);
        },

        //
        // Init editor callback
        //
        initEditorCallback: function(win, beforeStoreEffectsCallback) {
            var self = this;
            var effectArray = [];


            // Set max width and max height if image is smaller
            if (self.targetFormat && self.targetFormat.Width && self.originalMedia.Width < self.targetFormat.Width)
                self.targetFormat.Width = self.originalMedia.Width;

            if (self.targetFormat && self.targetFormat.Height && self.originalMedia.Height < self.targetFormat.Height)
                self.targetFormat.Height = self.originalMedia.Height;

            // Add effects from format if exists or create one effect from format width and height
            if (self.format.Effects !== undefined) {

                var cropZoomEffects = self.format.Effects.filter(function (e) {
                    return e.$type.indexOf("ImageVault.Common.Data.Effects.CropEffect,") > -1 ||
                        e.$type.indexOf("ImageVault.Common.Data.Effects.ResizeEffect,") > -1;
                });

                if (cropZoomEffects.length) {
                    effectArray = self.format.Effects;
                } else {
                    var width = self.format.Width;
                    var height = self.format.Height;
                    var resizeMode = 0; //ResizeToFit
                    var ar = self.format.AspectRatio;
                    // If aspect ratio is set and at least width or height, ResizeMode is ResizeToFill
                    if (ar && (width || height)) {
                        resizeMode = 1;
                        if (!width) {
                            width = ar * height;
                        } else if (!height) {
                            height = width / ar;
                        } else {
                            var far = width / height;
                            if (far > ar) {
                                width = ar * height;
                            } else {
                                height = width / ar;
                            }
                        }
                    }

                    // Add Resize effect and then crop effect to get selection in center of image
                    effectArray.push({
                        $type: "ImageVault.Common.Data.Effects.ResizeEffect,ImageVault.Common",
                        Width: Math.round(width),
                        Height: Math.round(height),
                        ResizeMode: resizeMode
                    });

                    // Add any non-crop-zoom effects as well
                    while (self.format.Effects.length) effectArray.push(self.format.Effects.pop());
                }
            }

            var effects = JSON.stringify(effectArray);

            var requestedVideoOrOriginal = self.format.$type !== "ImageVault.Common.Data.ImageFormat, ImageVault.Common";

            var rendererData = self.originalMedia.RendererData || {};
            if (typeof rendererData === 'string') {
                try {
                    rendererData = JSON.parse(rendererData);
                } catch (ex) {
                    rendererData = {};
                }
            }
            var originalIsVideo = self.originalMedia.ContentType.indexOf("video/") !== -1 ||
                self.originalMedia.RendererId === "BF2B805A-597A-4D46-87CA-90F3C9B9B50C"; //Azure Media Service player

            if (self.originalMedia.RendererId === "BF2B805A-597A-4D46-87CA-90F3C9B9B50C") {
                //set default keyFrameInterval to 2
                if (!rendererData.keyFrameInterval) {
                    rendererData["keyFrameInterval"] = 2;
                }
                //update rendererData with duration
                if (!rendererData.duration) {
                    rendererData["duration"] = self.duration;
                }
            }

            // Parameters for init editor
            var params = {
                id: self.originalMedia.Id,
                rendererId: self.originalMedia.RendererId,
                rendererData: rendererData,

                // Settings regarding the editor window
                editorSettings: { width: win.innerWidth, height: win.innerHeight },

                // Settings for communicating with core
                coreSettings: {
                    client: self.client,
                    mediaUrlBase: window.location.origin
                },

                // Translatable fields
                lang: self.initConfig.lang,
                languageSettings: {
                    editorWidthInputLabel: translate("externallink.EditorWidthInputLabel"),
                    editorHeightInputLabel: translate("externallink.EditorHeightInputLabel"),
                    editorOkButtonText: self.initConfig.editorOkButtonText ? self.initConfig.editorOkButtonText : translate("externallink.PublishButton"),
                    editorCancelButtonText: translate("externallink.EditorCancelButtonText"),
                    editorWarningIconText: translate("externallink.EditorWarningIconText"),
                    editorReloadButtonText: translate("externallink.EditorReloadButtonText"),
                    editorLoadErrorText: translate("externallink.EditorLoadErrorText"),
                    title: originalIsVideo && requestedVideoOrOriginal ? translate("externallink.PublishVideoTitle") : translate("externallink.PublishTitle")
                },
                // Current effects applied on the image
                currentEffects: effects,

                // End format (can be null)
                targetFormat: self.initConfig.targetFormatId ? {
                    width: self.targetFormat.Width,
                    height: self.targetFormat.Height,
                    resizeMode: self.targetFormat.AspectRatio ? 1 : 0,
                    CompressionQuality: self.targetFormat.CompressionQuality,
                    Dpi: self.targetFormat.Dpi
                } : null,

                // Original image format
                originalFormat: {
                    width: self.originalMedia.Width,
                    height: self.originalMedia.Height,
                    url: self.originalMedia.Url
                },
                // Set original format to image/jpeg if video that is requested in other format than original to get the correct editor (image editor)
                originalFormatContentType: !requestedVideoOrOriginal && originalIsVideo ? "image/jpeg" : self.originalMedia.ContentType,
                originalFormatUrl: self.originalMedia.Url,
                areaOfInterest: self.aoi,
                // Return an array of effects that should be applied to the media
                callback: function(effects) {
                    if (!effects || effects.length <= 0) {
                        beforeStoreEffectsCallback();
                        win.close();
                        $.event.trigger("/imagevault/propertymediacommon/editorclosed");
                        if (self.initConfig.callback)
                            self.initConfig.callback(null);

                        return;
                    }

                    // If callback is passed to editor as param, use this
                    if (self.initConfig.callback) {
                        self.initConfig.callback(effects);

                        return;
                    }

                    var externalLinkModel = {
                        Effects: effects,
                        MediaId: self.originalMedia.Id,
                        InputFormat: self.format,
                        MediaUrlBase: window.location.origin
                    };

                    // Show dialog with link and embed code to media
                    new ImageVaultUi.ExternalLinkDialog({
                        externalLinkModel: externalLinkModel,
                        client: self.client
                    }).open();

                    var externalLinkDialog = void 0;
                    $("body").append($(externalLinkDialog));

                    // Set focus on window
                    $(window).focus();
                },
                keepAspectRatio: self.targetFormat && self.targetFormat.Width && self.targetFormat.Height ? true : false,
                resizeMode: self.format.AspectRatio ? 1 : 0
            };

            // Open editor instance
            if (win.ImageVault.Editor.openEditorInstance) {
                win.ImageVault.Editor.openEditorInstance(params);
            }
        },

        //
        // Get size of window
        //
        getWindowInnerSize: function() {
            var w = window,
                d = document,
                e = d.documentElement,
                g = d.getElementsByTagName("body")[0],
                x = w.innerWidth || e.clientWidth || g.clientWidth,
                y = w.innerHeight || e.clientHeight || g.clientHeight;
            return {
                width: x,
                height: y
            };
        },
        getMetadata: function(mediaItem, name) {
            if (!mediaItem || !mediaItem.Metadata || !mediaItem.Metadata.length) return null;
            for (var i = 0; i < mediaItem.Metadata.length; i++) {
                var md = mediaItem.Metadata[i];
                if (md && md.Name === name) return md.Value;
            }
            return null;
        },
        //
        // Get media to update
        //
        getMedia: function(mediaId, formatId) {
            var self = this;
            // Parameters for webservice call "MediaService/Find"
            var mediaParameters = {
                Filter: { Id: [mediaId] },
                Populate: {
                    Metadata: [{ Filter: { Name: "areaOfInterest", MetadataDefinitionType: 1, isNull: false } },
                        { Filter: { Name: "duration", MetadataDefinitionType: 1, isNull: false } }
                    ],
                    MediaFormats: [{
                            // Original format (webmedia)
                            $type: "ImageVault.Common.Data.WebMediaFormat,ImageVault.Common"
                        },
                        {
                            // PDF/EPS cannot be converted to webmedia but fallback is possible to image format
                            $type: "ImageVault.Common.Data.ImageFormat,ImageVault.Common"
                        }
                    ]
                }
            };
            var findMediaJson = JSON.stringify(mediaParameters);

            app.json({
                url: 'api/mediaservice/find',
                data: findMediaJson,
                success: function (x) {
                    if (!x) {
                        alert("Error calling ImageVault Core. [" +
                            self.client._lastError.message +
                            " (status:" +
                            self.client._lastError.status +
                            ")]");
                        return null;
                    }

                    // Get original media conversion (to get properties like width and height for example)
                    self.originalMedia = x[0].MediaConversions[0] || x[0].MediaConversions[1];
                    self.aoi = null;
                    self.duration = null;
                    var aoiString = self.getMetadata(x[0], "AreaOfInterest");
                    if (aoiString) self.aoi = JSON.parse(aoiString);
                    var durationString = self.getMetadata(x[0], "Duration");
                    if (durationString) self.duration = parseFloat(durationString);

                    // Get requested format
                    var formatParameters = {
                        Filter: {
                            Format: {
                                Id: formatId
                            }
                        }
                    };

                    var findFormatJson = JSON.stringify(formatParameters);
                    app.json({
                        url: 'api/mediaformatservice/find',
                        data: findFormatJson,
                        success: function (formats) {
                            if (!formats) {
                                return null;
                            }
                            var format = formats[0];

                            self.format = format;

                            // Get format if requested from query (In insert mode) to get right height and width in editor settings
                            if (self.initConfig.targetFormatId) {
                                // Parameters for webservice call "MediaService/Find"
                                var targetFormatParameters = {
                                    Filter: {
                                        Format: {
                                            Id: self.initConfig.targetFormatId
                                        }
                                    }
                                };
                                var targetFormatParametersJson = JSON.stringify(targetFormatParameters);

                                app.json({
                                    url: 'api/mediaformatservice/find',
                                    data: targetFormatParametersJson,
                                    success: function (targetFormats) {
                                        if (!targetFormats) {
                                            return null;
                                        }
                                        self.targetFormat = targetFormats[0];
                                        self.showEditor();
                                        return null;
                                    },
                                    error: function (err) {
                                    return app.showError('Error calling ImageVault core', `${err?.Message} ${err?.ExceptionMessage}`);
                                    }
                                });
                                return null;
                            } else {
                                self.showEditor();
                                return null;
                            }
                        },
                        error: function (err) {
                        return app.showError('Error downloading media formats', `${err?.Message} ${err?.ExceptionMessage}`);
                        }
                    });
                },
                error: function (err) {
                    return app.showError('Error calling ImageVault core', `${err?.Message} ${err?.ExceptionMessage}`);
                }
            });
        },

        showEditor: function() {
            var self = this;
            // Set width and height and check max values
            var winSize = self.getWindowInnerSize();
            if (winSize.width > 1400)
                winSize.width = 1400;
            if (winSize.height > 900)
                winSize.height = 900;

            // Create popup
            $.fancybox({
                type: "iframe",
                href: window.location.origin + "/Scripts/imagevault-client-script/editor.html?v=5.41.9.0",
                width: winSize.width,
                height: winSize.height,
                scrolling: "no",
                showCloseButton: false,
                overlayColor: "#555",
                overlayOpacity: 0.5,
                onComplete: function() {
                    var $iframe = $("#fancybox-content iframe");
                    $iframe.load(function() {
                        $iframe.unbind("load");
                        $iframe.unbind("keydown");
                        var win = $iframe.get(0).contentWindow;
                        self.initEditorCallback(win, function() {
                            $.fancybox.close();
                        });

                        if (self.loader)
                            self.loader.remove();

                        $("#fancybox-content").append($("<div class='step-content-place-holder'></div>"));
                    });
                },
                onClosed: function() {
                    $.event.trigger("/imagevault/propertymediacommon/editorclosed");
                    return false;
                }
            });
        }
    };

    ns.ImageVaultEditor = imageVaultEditor;

})(jQuery, window.ImageVaultUi = window.ImageVaultUi || {});
