(function ($, ns, undefined) {
  const categoriesTab = function (config) {
    this.init(config)
    return this
  }
  categoriesTab.prototype = {
    // properties
    nav: null,
    $domobject: undefined,
    previousCategoryState: null,
    // methods
    init: function (config) {
      $.extend(this, config)
      if (this.$domobject === undefined) { throw '$domobject is not defined in config' }

      const self = this
      // Selected selected categories
      self.selectCategoriesFromHash()

      // Select exact match
      if ($.bbq.getState('em') === 'true') { $('#toggleExtendedSearch').attr('checked', false) }

      // Category tree
      $('li span.expand', this.$domobject).live('click', function () {
        self.toggleCategoryChildren($(this))
        // Reset hight of viewport
        self.nav.initViewport()
      })
      // Event for category selection
      $('input.category-checkbox', this.$domobject).live('change', function () {
        const $item = $(this).closest('li')

        const categoryId = $item.attr('data-category-id')
        if ($(this).is(':checked')) {
          $item.addClass('active')
          self.addCategoryToHash(categoryId)
        } else {
          self.removeCategoryFromHash(categoryId)
          $item.removeClass('active')
        }
      })

      // Event for show exact match
      $('#toggleExtendedSearch', this.$domobject).live('change', function () {
        const $item = $(this)

        if ($item.is(':checked')) {
          $.bbq.pushState({ em: false })
        } else {
          $.bbq.pushState({ em: true })
        }
      })

      // Search categories
      $('#searchCategory', this.$domobject).live('click', function (e) {
        const $ul = $('.category-tree > ul')
        ImageVaultUi.CategorySearchHelper.searchCategories($ul, e.target.value, false)
        return false
      })

      // Search agreements by enter key
      $('#categorySearchTerm', this.$domobject).live('keyup', function (e) {
        const $ul = $('.category-tree > ul')
        ImageVaultUi.CategorySearchHelper.searchCategories($ul, e.target.value, false)
      })

      $(window).bind('hashchange', function (e) {
        // Nothing has happend (with categories)
        if (e.getState('categories') === self.previousCategoryState) {
          return
        }
        self.selectCategoriesFromHash()
        self.previousCategoryState = e.getState('categories')
      })
      // monitor changes to categories
      app.bind('category:added', this.refreshCategoryList, this)
      app.bind('category:deleted', this.refreshCategoryList, this)
      app.bind('category:edited', this.refreshCategoryList, this)
    },
    // called when the tab should be displayed
    show: function () {
      this.$domobject.show()
      app.unbind(ImageVaultUi.NavigationEvents.navtoolbarEditClick)
      app.trigger(ImageVaultUi.NavigationEvents.navtoolbarDisableButton, ImageVaultUi.NavigationEvents.navtoolButtons.edit)
    },
    // called when the tab should be hidden
    hide: function () {
      this.$domobject.hide()
    },

    /*************************************************
		 *** BBQ hash persistance methods
		*************************************************/
    // adds the category to the category hash
    addCategoryToHash: function (categoryId) {
      const categories = this.getSelectedCategories()
      // Add category id to array
      categories.push(categoryId)

      this.saveCategoriestToHash(categories)
    },
    removeCategoryFromHash: function (categoryId) {
      const categories = this.getSelectedCategories()
      // Remove category from array.
      for (let i = 0; i < categories.length; i++) {
        if (categories[i] == categoryId) {
          categories.splice(i, 1)
          break
        }
      }
      this.saveCategoriestToHash(categories)
    },
    saveCategoriestToHash: function (categories) {
      // Remove key if no categories left
      if (categories.length == 0) {
        $.bbq.removeState('categories')
        return
      }
      // Push state to hash
      const state = {}
      state.categories = categories.join(',')
      $.bbq.pushState(state)
    },
    // Gets selected categories from hash fragment and returns as array
    getSelectedCategories: function () {
      // using bbq
      const categoriesAsString = $.bbq.getState('categories')
      if (typeof categoriesAsString === 'undefined' || categoriesAsString.length == 0) {
        return []
      }
      return categoriesAsString.split(',')
    },

    toggleCategoryChildren: function ($element) {
      // Expand children
      if (!$element.hasClass('expanded')) {
        $element.addClass('expanded')
        $element.siblings('ul').show()
      } else {
        $element.removeClass('expanded')
        $element.siblings('ul').hide()
      }
    },

    selectCategoriesFromHash: function () {
      $('input.category-checkbox', this.$domobject).removeAttr('checked')
      $('#categories li').removeClass('active')
      const categories = this.getSelectedCategories()
      for (let i in categories) {
        const $listItem = $('li[data-category-id=' + categories[i] + ']', this.$domobject)
        const $element = $listItem.children().children('input.category-checkbox')

        $element.attr('checked', 'checked')
        $listItem.addClass('active')
      }
    },

    // refreshes the category list
    refreshCategoryList: function () {
      const self = this
      // store all expanded categories
      const expandedCategories = this.getExpandedCategories()
      const selectedCategories = this.getSelectedCategories()
      app.load({
        controller: 'library',
        action: 'RefreshCategoryList',
        success: function (data) {
          // update vault list
          $('div#categories>div.category-tree').html(data)
          // remove any categories from selection that was deleted
          const availableCategories = self.getAvailableCategories()
          const removedSelectedCategories = _.difference(selectedCategories, availableCategories)
          for (let i = 0; i < removedSelectedCategories.length; i++) {
            self.removeCategoryFromHash(removedSelectedCategories[i])
            app.log('removed selected category ' + removedSelectedCategories[i] + ' since it was deleted')
          }
          // restore expanded state
          self.expandCategories(expandedCategories)
          // select the categories that should be selected
          self.selectCategoriesFromHash()
        }
      })
    },
    // gets all available categories (id)
    getAvailableCategories: function () {
      return $('>ul li', this.$domobject).map(function () {
        return $(this).attr('data-category-id')
      }).get()
    },
    // gets all expanded categories(id)
    getExpandedCategories: function () {
      return $('ul ul:visible', this.$domobject).parent().map(function () {
        return $(this).attr('data-category-id')
      }).get()
    },
    // Expands the supplied categories (list of id)
    expandCategories: function (list) {
      for (let i = 0; i < list.length; i++) {
        this.expandCategory(list[i])
      }
    },
    // expands the selected category (id)
    expandCategory: function (categoryId) {
      const $element = $("li[data-category-id='" + categoryId + "']>span.expand")
      $element.addClass('expanded')
      $element.siblings('ul').show()
    }
  }
  ns.NavCategoriesTab = categoriesTab
})(jQuery, window.ImageVaultUi = window.ImageVaultUi || {})
