﻿(function($, ns, undefined) {
    var navlibraryTab = function(config) {
        this.init(config);
        return this;
    };
    navlibraryTab.prototype = {
        // Properties
        nav: null,
        $domobject: undefined,

        /**
         * Init function
         * @param {} config 
         * @returns {} 
         */
        init: function(config) {
            $.extend(this, config);
            if (this.$domobject === undefined)
                throw "$domobject is not defined in config";
            var self = this;

            app.bind("vaultDialog:vaultDeleted", this.vaultDeleted, this);
            app.bind("vaultDialog:vaultSaved", this.vaultSaved, this);
            app.bind("collectionDialog:collectionCreated", this.collectionCreated, this);
            app.bind("collectionDialog:collectionUpdated", this.collectionUpdated, this);
            app.bind("collectionDialog:collectionSaved", this.collectionSaved, this);
            app.bind("collectionDialog:collectionDeleted", this.collectionDeleted, this);

            //Event for vault selection
            $("input.vault-checkbox", this.$domobject).live("change", function() {
                var $item = $(this).closest("li");
                var vaultId = $item.attr("data-vault-id");

                // If only select one vault
                if ($.bbq.getState("tsmv") !== "true") {
                    // If same as existing remove selection
                    var currentSelectedVaults = self.getSelectedVaults();
                    if (currentSelectedVaults && vaultId === currentSelectedVaults[0])
                        self.saveVaultsToHash([]);
                    else
                        self.saveVaultsToHash([vaultId]);

                    self.selectVaultsFromHash();
                    return;
                }

                if ($(this).is(":checked")) {
                    if (vaultId === "0") {
                        var allVaults = $("#vaults li").map(function() {
                            return $(this).attr("data-vault-id");
                        }).get();

                        self.saveVaultsToHash(allVaults);
                    } else {
                        self.addVaultToHash(vaultId);
                    }
                } else {
                    if (vaultId === "0") {
                        self.saveVaultsToHash([]);
                    } else {
                        self.removeVaultFromHash(vaultId);
                    }
                }

                self.selectVaultsFromHash();
            });

            //Event for collection selection
            $("input.collection-checkbox", this.$domobject).live("change", function(evt) {
                self.selectCollection(evt.currentTarget);
            });

            // Toggle select multiple vaults
            $("#toggleSelectMultipleVaults", this.$domobject).live("change", function() {
                var $item = $(this);
                if ($item.is(":checked")) {
                    $.bbq.pushState({ tsmv: true });

                } else {
                    $.bbq.pushState({ tsmv: false });
                }
            });

            // Toggle select multiple collections
            $("#toggleSelectMultipleCollections", this.$domobject).live("change", function() {
                var $item = $(this);
                if ($item.is(":checked")) {
                    $.bbq.pushState({ tsmc: true });

                } else {
                    $.bbq.pushState({ tsmc: false });
                }
            });

            $(window).bind("hashchange", function(e) {
                self.updateFromHash();
                if (e.getState("tab") !== ImageVaultUi.NavigationEvents.tabs.library)
                    return;
                if (!app.hasStateChanged("tab") && !app.hasStateChanged("vaults") && !app.hasStateChanged("collectionid"))
                    return;
            });
            this.setupCollectionsDragAndDrop();
        },


        /**selectCollection
         * Selects a collection
         * @returns {}
         */
        selectCollection: function(elem) {
            var self = this;
            var $item = $(elem).closest("li");

            var collectionId = $item.attr("data-collection-id");

            // If only select one collection
            if ($.bbq.getState("tsmc") !== "true") {

                // Deselect all before selecting a collection
                // in order to refresh tags after a name change.
                self.saveCollectionsToHash([]);
                self.selectCollectionsFromHash();

                self.saveCollectionsToHash([collectionId]);
                self.selectCollectionsFromHash();

                return;
            }

            if ($(elem).is(":checked")) {
                if (collectionId === "0") {
                    self.saveCollectionsToHash($("#collections li").map(function() {
                        return $(elem).attr("data-collection-id");
                    }).get());
                } else {
                    self.addCollectionToHash(collectionId);
                }
            } else {
                if (collectionId === "0") {
                    self.saveCollectionsToHash([]);
                } else {
                    self.removeCollectionFromHash(collectionId);
                }
            }
        },

        /**selectCollectionsFromHash
         * Update selected values from hash
         * @returns {} 
         */
        updateFromHash: function() {

            if (!$.bbq.getState("tab") || $.bbq.getState("tab").indexOf(ImageVaultUi.NavigationEvents.tabs.library))
                return;


            // First unbind click on edit button
            app.unbind(ImageVaultUi.NavigationEvents.navtoolbarEditClick);

            // Select select multiple vaults
            if ($.bbq.getState("tsmv") === "true") {
                $("#toggleSelectMultipleVaults").attr("checked", "checked");
                $("#vaults").addClass("select-multiple");
            } else {
                $("#toggleSelectMultipleVaults").removeAttr("checked");
                $("#vaults").removeClass("select-multiple");
            }
            // Select select multiple vaults
            if ($.bbq.getState("tsmc") === "true") {
                $("#toggleSelectMultipleCollections").attr("checked", "checked");
                $("#collections").addClass("select-multiple");
            } else {
                $("#toggleSelectMultipleCollections").removeAttr("checked");
                $("#collections").removeClass("select-multiple");
            }

            // Select vaults from hash (exclude "all" vault (0))
            var selectedVaults = this.selectVaultsFromHash().filter(function(i) { return i != 0; });
            // Select collections from hash (exclude "all" collection (0))
            var selectedCollections = this.selectCollectionsFromHash().filter(function(i) { return i != 0; });

            // If only one option chosen, activate edit button
            if ((selectedVaults.length + selectedCollections.length) === 1) {
                // if the one item is a selectedCollection or a vault which the user has admin access to (collections are automatically admins since only admins can see them)
                if (selectedCollections.length || (selectedVaults.length && this.getSelectedVaultIsAdmin())) {
                    app.trigger(ImageVaultUi.NavigationEvents.navtoolbarEnableButton, ImageVaultUi.NavigationEvents.navtoolButtons.edit);
                } else {
                    app.trigger(ImageVaultUi.NavigationEvents.navtoolbarDisableButton, ImageVaultUi.NavigationEvents.navtoolButtons.edit);
                }
                // point to right dialog
                if (selectedVaults.length)
                    app.bind(ImageVaultUi.NavigationEvents.navtoolbarEditClick, this.openEditDialogForSelectedVault, this);
                else
                    app.bind(ImageVaultUi.NavigationEvents.navtoolbarEditClick, this.openEditDialogForSelectedCollection, this);
            } else {
                // set state of delete button
                app.trigger(ImageVaultUi.NavigationEvents.navtoolbarDeleteState, ImageVaultUi.NavigationEvents.navtoolbarDeleteStates.delete);
                app.unbind(ImageVaultUi.NavigationEvents.navtoolbarEditClick);
                app.trigger(ImageVaultUi.NavigationEvents.navtoolbarDisableButton, ImageVaultUi.NavigationEvents.navtoolButtons.edit);
            }

            if (selectedCollections.length) {
                app.trigger(ImageVaultUi.NavigationEvents.navtoolbarDeleteState, ImageVaultUi.NavigationEvents.navtoolbarDeleteStates.removeFromCollection);
            } else {
                app.trigger(ImageVaultUi.NavigationEvents.navtoolbarDeleteState, ImageVaultUi.NavigationEvents.navtoolbarDeleteStates.delete);
            }
        },

        /**
         * Select vaults from hash
         * @returns {} 
         */
        selectVaultsFromHash: function() {
            $("input.vault-checkbox", this.$domobject).removeAttr("checked");
            $("#vaults li").removeClass("active");

            var vaultIds = this.getSelectedVaults();
            for (var i = 0; i < vaultIds.length; i++) {

                var $listItem = $("li[data-vault-id=" + vaultIds[i] + "]");
                var $element = $listItem.find("input.vault-checkbox");

                $element.attr("checked", "checked");
                $listItem.addClass("active");

            }

            if (vaultIds.length === 0) {
                $("li[data-vault-id=0]").addClass("active");
                //$("li[data-vault-id=0] input.vault-checkbox").attr("checked", "checked");
            }
            return vaultIds;
        },

        /**
         * Select collections from hash
         * @returns {} 
         */
        selectCollectionsFromHash: function() {
            $("input.collection-checkbox", this.$domobject).removeAttr("checked");
            $("#collections li").removeClass("active");

            var collectionIds = this.getSelectedCollections();
            for (var i in collectionIds) {

                var $listItem = $("li[data-collection-id=" + collectionIds[i] + "]", this.$domobject);
                var $element = $listItem.children().children("input.collection-checkbox");

                $element.attr("checked", "checked");
                $listItem.addClass("active");

            }

            if (collectionIds.length === 0) {
                //$("li[data-collection-id=0]").addClass("active");
                //$("li[data-collection-id=0] input.vault-checkbox").attr("checked", "checked");
            }

            return collectionIds;
        },

        /**
         * Get selected vaults
         * @returns {} 
         */
        getSelectedVaults: function() {
            //using bbq
            var vaultsAsString = $.bbq.getState("vaults");
            if (typeof vaultsAsString === "undefined" || vaultsAsString.length === 0) {
                return new Array();
            }
            return vaultsAsString.split(",");
        },

        /**
         * Get selected collections
         * @returns {} 
         */
        getSelectedCollections: function() {
            //using bbq
            var collectionsAsString = $.bbq.getState("collections");
            if (typeof collectionsAsString === "undefined" || collectionsAsString.length === 0) {
                return new Array();
            }
            return collectionsAsString.split(",");
        },

        /**
         * Add vault to hash
         * @param {} vaultId 
         * @returns {} 
         */
        addVaultToHash: function(vaultId) {
            var vaults = this.getSelectedVaults();

            //Add vault id to array
            vaults.push(vaultId);

            this.saveVaultsToHash(vaults);
        },

        /**
         * Add collection to hash
         * @param {} collectionId
         * @returns {} 
         */
        addCollectionToHash: function(collectionId) {
            var collections = this.getSelectedCollections();

            //Add collection id to array
            collections.push(collectionId);

            this.saveCollectionsToHash(collections);
        },

        /**
         * Remove vault from hash
         * @param {} vaultId 
         * @returns {} 
         */
        removeVaultFromHash: function(vaultId) {
            var vaults = this.getSelectedVaults();

            //Remove vault from array.
            for (var i = 0; i < vaults.length; i++) {
                if (vaults[i] === vaultId) {
                    vaults.splice(i, 1);
                    break;
                }
            }
            this.saveVaultsToHash(vaults);
        },

        /**
         * Remove collection from hash
         * @param {} collectionId
         * @returns {} 
         */
        removeCollectionFromHash: function(collectionId) {
            var collections = this.getSelectedCollections();

            // Remove collection from array.
            for (var i = 0; i < collections.length; i++) {
                if (collections[i] === collectionId) {
                    collections.splice(i, 1);
                    break;
                }
            }
            this.saveCollectionsToHash(collections);
        },

        /**
         * Save vaults to hash
         * @param {} vaults 
         * @returns {} 
         */
        saveVaultsToHash: function(vaults) {
            //Remove key if no vaults left
            if (vaults.length === 0) {
                $.bbq.removeState("vaults");
                return;
            }

            //Push state to hash
            var state = {};
            state["vaults"] = vaults.join(",");
            $.bbq.pushState(state);
        },

        /**
         * Save collections to hash
         * @param {} collections 
         * @returns {} 
         */
        saveCollectionsToHash: function(collections) {
            //Remove key if no collections left
            if (collections.length === 0) {
                $.bbq.removeState("collections");
                return;
            }

            //Push state to hash
            var state = {};
            state["collections"] = collections.join(",");
            $.bbq.pushState(state);
        },

        /**
         * shows the library tab
         * @returns {} 
         */
        show: function() {
            this.$domobject.show();

            this.updateFromHash();
        },

        /**
         * hides the library tab
         * @returns {} 
         */
        hide: function() {
            this.$domobject.hide();
        },

        /**
         * opens the vault dialog for the current selected vault
         * @returns {} 
         */
        openEditDialogForSelectedVault: function() {
            var vaultId = this.getSelectedVaults()[0];
            ImageVaultUi.VaultDialog.OpenDialog(vaultId);
        },

        /**
         * opens the collection dialog for the selected collection
         * @returns {} 
         */
        openEditDialogForSelectedCollection: function() {
            var collectionId = this.getSelectedCollections()[0];
            new ImageVaultUi.CollectionDialog({ collectionId: collectionId }).open();
        },

        /**
         * opens the new vault dialog
         * @returns {} 
         */
        openNewVaultDialog: function() {
            ImageVaultUi.VaultDialog.OpenDialog(-1);
        },

        /**
         * callback methods for vault getting saved
         * @param {} vault 
         * @returns {} 
         */
        vaultSaved: function(vault) {
            this.refreshAndSelectVault(false, vault.Id);
        },

        /**
         * callback method for vault getting deleted
         * @param {} vaultId 
         * @returns {} 
         */
        vaultDeleted: function(vaultId) {
            this.refreshAndSelectVault(true, "0");
        },

        /**
         * checks if the user is admin for the selected vault
         * @returns {} 
         */
        getSelectedVaultIsAdmin: function() {
            if (!$("li.active", this.$domobject).attr("data-is-vault-admin")) {
                return false;
            }
            return $("li.active", this.$domobject).attr("data-is-vault-admin") === "True";
        },

        /**
         * Refreshes the vault list and selects the vault with the given id.
         * @param {} vaultId 
         * @returns {} 
         */
        refreshAndSelectVault: function(selectAllVaults, vaultId) {
            var self = this;
            app.load({
                controller: "library",
                action: "RefreshVaultList/" + vaultId,
                success: function(data) {
                    //force select the vaults tab
                    $.bbq.pushState({ tab: ImageVaultUi.NavigationEvents.tabs.library });
                    //update vault list
                    $("#vault-list", self.$domobject).html(data);

                    if (selectAllVaults) {
                        $("#stream").empty();
                        setTimeout(function() {
                            $("#vaults li[data-vault-id='0'] input", self.$domobject).click();
                        }, 1000);
                    } else {
                        // deselect and reselect vault to update scope (in case of edit)
                        $("#vaults li[data-vault-id='" + vaultId + "'] input", self.$domobject).click();
                        $("#vaults li[data-vault-id='" + vaultId + "'] input", self.$domobject).click();
                    }
                }
            });
        },

        /**
         * Triggers when collection is saved
         * @param {} id 
         * @returns {} 
         */
        collectionCreated: function(id) {
            this.refreshCollections(true, id);
        },
        /**
         * Triggers when collection is saved
         * @param {} id 
         * @returns {} 
         */
        collectionUpdated: function(id) {
            this.refreshCollections(false, id);
        },
        /**
         * Triggers when collection is saved
         * @param {} id 
         * @returns {} 
         */
        collectionSaved: function(id) {
            //ToDo:
        },

        /**
         * Triggers when collection is deleted
         * @param {} id 
         * @returns {} 
         */
        collectionDeleted: function(id) {
            this.refreshCollections(true);
        },

        /**
         * Refresh collections list
         * @param {} selectAllVaults 
         * @returns {} 
         */
        refreshCollections: function(selectAllCollections, collectionId) {
            var self = this;
            app.load({
                controller: "library",
                action: "RefreshCollectionList",
                success: function(data) {
                    // force select the vaults tab
                    $.bbq.pushState({ tab: ImageVaultUi.NavigationEvents.tabs.library });
                    // update collection list
                    $("#collection-list", self.$domobject).html(data);
                    // reinitialize scrollbars
                    // $(window).trigger("reinitializenavscroll");
                    // set up drag n drop for new collection list
                    self.setupCollectionsDragAndDrop();
                    if (selectAllCollections) {
                        self.selectCollection($("#collections li[data-collection-id='0'] input", self.$domobject)[0]);
                    } else {
                        self.selectCollection($("#collections li[data-collection-id='" + collectionId + "'] input", self.$domobject)[0]);
                    }


                }
            });
        },

        /**
         * Add media item to collection
         * @param {} e 
         * @returns {} 
         */
        addToCollection: function(e) {
            var collectionId = $(e.target).closest("li").attr("data-collection-id");

            var mediaItemIds = $.bbq.getState("items");

            if (mediaItemIds != undefined) {
                mediaItemIds = mediaItemIds.split(",");
            } else {
                mediaItemIds = null;
            }

            var loading = new IVNotification(translate("common.Loading"));

            app.load({
                type: "POST",
                controller: "Collection",
                action: "AddToCollection",
                data: { collectionId: collectionId, mediaItemIds: mediaItemIds },
                success: function(data) {
                    app.trigger("collection:itemAdded", collectionId);
                    loading.remove();
                }
            });
        },

        /**
         * Setup drag and drop
         * @returns {} 
         */
        setupCollectionsDragAndDrop: function() {
            var self = this;
            var $dropTargets = $("#collection-list li");

            $dropTargets.droppable({
                scope: "mediaItem",
                accept: "li",
                hoverClass: "dragover",
                tolerance: "pointer",
                over: function() {
                    $(".drag-items-helper").addClass("over");
                    $(".drag-items-helper").append('<img src="images/icon-add.png" class="helper" />');
                },
                out: function() {
                    $(".drag-items-helper").removeClass("over");
                    $(".drag-items-helper").find("img:last-child").remove();
                },
                drop: function(e, ui) {
                    self.addToCollection(e);
                }
            });
        }
    };
    ns.NavLibraryTab = navlibraryTab;
})(jQuery, window.ImageVaultUi = window.ImageVaultUi || {});